#!/usr/bin/env bash
#
# ====================[ install_linux.sh                  ]====================
#
# --------------------( SYNOPSIS                          )--------------------
# Linux-based installer for the "Long War Rebalance" XCOM mod.
#
# --------------------( INSTALLATION                      )--------------------
# Copy this script into the top-level directory to which you unzipped the
# contents of the "Long War Rebalance" zip archive and run that script to
# install "Long War Rebalance": e.g.,
#
# 1. Download both this installer and the "Long War Rebalance" zip archive into
#    the current directory.
# 2. Unzip this archive:
#    unzip 'LW Rebalance '*.zip
# 3. Copy this installer into the unzipped directory:
#    cp install_linux.bash 'LW Rebalance '*/
# 4. Run this script to install "Long War Rebalance":
#    bash 'LW Rebalance '*/install_linux.bash
#
# --------------------( USAGE                             )--------------------
# $ bash install_linux.bash [XCOM_INSTALL_DIRECTORY]
#
# This script optionally accepts the absolute or relative dirname of the
# top-level installation directory for "XCOM: Enemy Unknown." When unpassed,
# this argument defaults to the typical Steam installation directory (i.e.,
# "~/.steam/steam/steamapps/common/XCom-Enemy-Unknown").
#
# --------------------( EXAMPLES                          )--------------------
# * Install "Long War Rebalance" to the typical Steam installation directory:
#     bash install_linux.bash
# * Install "Long War Rebalance" to a custom installation directory:
#     bash install_linux.bash ~/Games/XCom_Enemy_Within
#
# --------------------( AUTHORS                           )--------------------
# * Original author:
#     https://github.com/leycec
#
# --------------------( SEE ALSO                          )--------------------
# * Nexus forum post inspiring this script:
#     https://forums.nexusmods.com/index.php?showtopic=6697482/#entry64622831

# ....................{ PREAMBLE                          }....................
# Enforce strictness.
set -e

# If the user passed more than one argument, print an error and fail.
if (( $# > 1 )); then
    echo 'Expected no arguments or one target dirname.' >&2
    exit 1
fi

# Absolute canonical dirname of the directory containing this script. Note the
# "readlink -f" option requires the macOS-incompatible GNU coreutils.
SRC_DIRNAME="$(dirname "$(readlink -f "${BASH_SOURCE[0]}")")"

# Absolute or relative dirname of the top-level XCOM installation directory,
# defaulting to the typical Steam installation directory if unpassed.
TRG_DIRNAME="${1:-${HOME}/.steam/steam/steamapps/common/XCom-Enemy-Unknown}"

# Print a welcome message.
echo "Installing Long War Rebalance: ${SRC_DIRNAME} -> ${TRG_DIRNAME}"

# If this installation directory does *NOT* exist, print an error and fail.
if [[ ! -d "${TRG_DIRNAME}" ]]; then
    echo "Target directory \"${TRG_DIRNAME}\" not found." >&2
    exit 1
fi

# ....................{ RENAME                            }....................
# Coerce all filenames in the "CookedPCConsole" and "Localization"
# subdirectories to lowercase. Note that this does *NOT* apply to the "Config"
# subdirectory, whose filenames are intentionally preserved as is.
find "${SRC_DIRNAME}/CookedPCConsole/" "${SRC_DIRNAME}/Localization/" \
    -depth -type f | while read -r SRC_FILENAME; do
    # Absolute filename of this file coerced to lowercase.
    TRG_FILENAME="$(dirname "${SRC_FILENAME}")/$(basename "${SRC_FILENAME}" |
        tr '[:upper:]' '[:lower:]')"

    # If the two filenames differ (i.e., the original filename contains at
    # least one uppercase character), coerce this filename to lowercase.
    if [[ "${SRC_FILENAME}" != "${TRG_FILENAME}" ]]; then
        mv -v -T "${SRC_FILENAME}" "${TRG_FILENAME}"
    fi
done

# Rename the "Default" prefix of every filename in the "Config" subdirectory to
# "XCom" instead (e.g., "Config/DefaultMaps.ini" to "Config/XComMaps.ini").
for SRC_CONFIG_FILENAME in "${SRC_DIRNAME}"/Config/*; do
    # Basename of this file with this prefix replaced.
    TRG_CONFIG_BASENAME="$(basename "${SRC_CONFIG_FILENAME}")"
    TRG_CONFIG_BASENAME="XCom${TRG_CONFIG_BASENAME#Default}"

    # Absolute filename of this file with this prefix replaced.
    TRG_CONFIG_FILENAME="$(dirname "${SRC_CONFIG_FILENAME}")/${TRG_CONFIG_BASENAME}"

    # Rename this prefix in this filename.
    mv -v -T "${SRC_CONFIG_FILENAME}" "${TRG_CONFIG_FILENAME}"
done

# ....................{ INSTALL                           }....................
# Install all files in the "Config" subdirectory.
cp -v "${SRC_DIRNAME}"/Config/* \
    ~/.local/share/feral-interactive/XCOM/XEW/WritableFiles/

# Install all files in the "CookedPCConsole" subdirectory.
cp -v "${SRC_DIRNAME}"/CookedPCConsole/* \
    "${TRG_DIRNAME}"/xew/xcomgame/cookedpcconsole/

# Install all files in the "Localization/INT" subdirectory.
cp -v "${SRC_DIRNAME}"/Localization/INT/* \
    "${TRG_DIRNAME}"/xew/binaries/share/feraloverrides/
cp -v "${SRC_DIRNAME}"/Localization/INT/* \
    "${TRG_DIRNAME}"/xew/xcomgame/localization/int/

# ....................{ POSTAMBLE                         }....................
# Print a shutdown message.
echo 'Installed!'